' Penrose P2 Tilings
' Rev 1.0.0 William M Leue 4/26/2021

option default integer
option base 1
option angle degrees

const S1 = 50
const A1 = 72
const A2 = 144
const A3 = 36
const G = (1.0+sqr(5))/2.0
const IG = 1.0/G
const KITE = 1
const DART = 2

const STAR = 1
const ACE = 2
const SUN = 3
const KING = 4
const JACK = 5
const QUEEN = 6
const DEUCE = 7

const NUM_VERTICES = 7
const MAX_VTILES = 5
const NUM_VINFO = 5
const MAX_TILES = 500
const FSIDE = 6

const UP = 128
const DOWN = 129
const LEFT = 130
const RIGHT = 131
const QUIT = 81
const QUITL = 113

dim colors(5) = (rgb(RED), rgb(0,150,0), rgb(0, 0, 150), rgb(YELLOW), rgb(GRAY))
dim float vinfo(NUM_VERTICES, MAX_VTILES, NUM_VINFO)
dim vnum(NUM_VERTICES)
dim vnames$(NUM_VERTICES) = ("Star", "Ace", "Sun", "King", "Jack", "Queen", "Deuce")
dim corner_distance(2, 3)
dim size = S1
dim num_tiles = 0
dim tiles(MAX_TILES, NUM_VINFO)
dim dtiles(MAX_TILES, NUM_VINFO)
dim num_dtiles = 0
dim nsize = size/G

open "debug.txt" for output as #1
ReadVertices
ShowHelp
RunProgramWithParams "Penrose", "1"
end

sub ShowHelp
  ShowHelp1
  ShowHelp2
  ShowHelp3
  ShowHelp4
  ShowHelp5
  ShowHelp6
end sub

sub RunProgramWithParams name$, par$
  local prg$
  prg$ = name$ + ".bas"
  Execute("Run " + chr$(34) + prg$ + chr$(34) + "," + par$)
end sub

sub ShowHelp1
  local z$ = INKEY$
  cls
  text mm.hres\2, 0, "Penrose P2 Tilings Help", "CT", 4,, rgb(green)
  print @(0, 50)
  print "The P2 tiles were discovered by physicist Roger Penrose in 1975. These two tiles,"
  print "named the 'Kite' and 'Dart' tile a plane only aperiodically if the matching rules"
  print "are observed. Penrose also discovered the P3 tiles, which the 'thick' and 'thin'"
  print "rhombs. Both sets of tiles have strong ties to the golden ratio phi."
  print ""
  print "A periodic tiling of the plane is one that can be slid in some direction and"
  print "superimposed on itself to create an identical tiling. An aperiodic tiling is one"
  print "where this kind of duplication is impossible."
  print ""
  print "The Kite and Dart tiles each have 4 sides; two short and two long. The ratio of the"
  print "long to short sides is the golden ratio phi: 1.618033.. All the angles at the tile"
  print "corners are multiples of 36 degrees.
  print ""
  text mm.hres\2, 580, "Press Any Key to Continue", "CT"
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

sub ShowHelp2
  local z$ = INKEY$
  local m$
  cls
  text mm.hres\2, 0, "Penrose Kite and Dart Tiles", "CT", 4,, rgb(green)
  DrawP2Tile KITE, 200, 200, 200, 5, 1, rgb(white), rgb(yellow)
  DrawP2Tile DART, 500, 200, 200, 5, 1, rgb(white), rgb(200, 100, 0)
  text 200, 150, "Kite", "CT", 4,, rgb(green)
  text 500, 150, "Dart", "CT", 4,, rgb(green)
  m$ = "72" + chr$(96)
  text 200, 220, m$, "CM", 7,, rgb(black), -1
  text 100, 360, m$, "CM", 7,, rgb(black), -1
  text 300, 360, m$, "CM", 7,, rgb(black), -1
  text 500, 220, m$, "CM", 7,, rgb(black), -1
  m$ = "144" + chr$(96)
  text 200, 390, m$, "CM", 7,, rgb(black), -1
  m$ = "36" + chr$(96)
  text 410, 345, m$, "CM", 7,, rgb(black), -1
  text 590, 345, m$, "CM", 7,, rgb(black), -1
  m$ = "216" + chr$(96)
  text 500, 320, m$, "CM", 7,, rgb(black), -1
  print @(0, 430) ""
  print "Let's say the short sides of each tile have length 'S'. Then the long sides of"
  print "each tile have length S*phi, where phi = (1 + sqrt(5))/2. [The Golden Ratio]

  text mm.hres\2, 580, "Press Any Key to Continue", "CT"
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

sub ShowHelp3
  cls
  local z$ = INKEY$
  text mm.hres\2, 0, "Matching Rules and Possible Filled Vertices", "CT", 4,, rgb(green)
  print @(0, 50) ""
  print "When tiling with the Penrose P2 tiles, adjacent sides of two tiles must of course"
  print "be the same length. There are other restrictions. Don't worry about them now,"
  print "because this program will automatically enforce them."
  print ""
  print "With these matching rules enforced, there are exactly seven unique ways to arrange"
  print "combinations of the Kite and Dart tiles around a vertex in the tiling so that the"
  print "vertex is completely filled and no tiles overlap.
  print ""
  print "These vertices have been given names:
  print "  Sun:   Five Kite tiles with their sharp angles touching the vertex."
  print "  Star:  Five Dart tiles with their shart angles touching the vertex."
  print "  King:  Three Kite tiles and two Dart tiles,"
  print "  Queen: Four Kite tiles and one Dart tile."
  print "  Jack:  Three Kite tile and two Dart tiles (different than King.)"
  print "  Deuce: Two Kite tiles and two Dart tiles."
  print "  Ace:   One Kite tile and two Dart tile."
  print ""
  print ""
  text mm.hres\2, 580, "Press Any Key to Continue", "CT"
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

sub ShowHelp4
  cls
  local z$ = INKEY$
  DrawVertices
  text mm.hres\2, 580, "Press Any Key to Continue", "CT"
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

sub ShowHelp5
  cls
  local z$ = INKEY$
  text mm.hres\2, 0, "Tiling by Deflation", "CT", 4,, rgb(green)
  print @(0, 50) ""
  print "Putting together a tiling manually can be tedious and error-prone."
  print "Even if the matching rules are followed when laying down each tile, you can find"
  print "that you have created a space that cannot be tiled."
  print ""
  print "Fortunately, there is an easy and quick way to tile a large area with hundreds or"
  print "thousands of tiles automatically. This method is called DEFLATION, and it can be used"
  print "with both Penrose P2 and P3 tiles."
  print ""
  print "In deflation, each Kite and Dart tile is partitioned into several tiles, each somewhat"
  print "smaller than the original tile. This process can be continued recursively, successively"
  print "creating more and more smaller and smaller tiles.
  print ""
  print "In deflation, each original Kite tile gets converted into two smaller Kites and two"
  print "smaller Darts. Actually, that is not quite accurate. In fact, each Kite gets converted"
  print "into two smaller HALF Kites and two smaller HALF Darts. The other halves of these"
  print "'daughter' tiles come from the neighbors of the original Kite.
  print ""
  print "Similarly, each original Dart tile gets converted into one half-Kite and two half-darts,"
  print "with the other half of these new tiles coming from the Dart's neighbor tiles."
  print ""
  print "This process is hard to visualize when you just have a verbal description, but it is"
  print "easy to understand when you see it working."
  print ""
  print "[Implementation Note]"
  print "Different algorithms can be chosen to implement deflation. I have chosen to use an"
  print "iterative rather than a recursive algorithm to optimize speed. This version also creates"
  print "many redundant tiles that are clones of the ones you see. These extra tiles could be"
  print "removed, but it is faster to just leave them alone than try to remove them,"
  text mm.hres\2, 580, "Press Any Key to Continue", "CT"
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

sub ShowHelp6
  cls
  local z$ = INKEY$
  text mm.hres\2, 0, "Commands for Making Penrose Tilings", "CT", 4,, rgb(green)
  print @(0, 50) ""
  print "On the START SCREEN, you choose one of the 7 unique tiling vertices, then choose"
  print "a tile size and a color scheme. Using the UP and DOWN arrow keys to move from"
  print "one choice selection to another. Use the LEFT and RIGHT arrow keys to choose a value"
  print "for the current selection. Press ENTER to start tiling."
  print ""
  print "There are two tile sizes, NORMAL and LARGE. Normal is chosen so that all tiles will"
  print "be visible in all deflations. LARGE is chosen so that tne entire screen will be filled."
  print ""
  print "There are 4 color schemes:
  print "  1. Orange Kites and Yellow Darts [the default]"
  print "  2. Green Kites and Blue Darts"
  print "  3. Tile colors vary by tile rotation angle: Kites in red part of spectrum, Darts in blue."
  print "  4. White Tiles with Black borders." 
  print ""
  print "Once you have pressed ENTER, your chosen vertex appears."
  print ""
  print "Next, press the SPACEBAR to deflate the starting vertex by one step. You will see"
  print "more tiles appear, each smaller than the previous tiles. Repeat the deflation"
  print "again by pressing the SPACEBAR again. At each step, there will be more tiles."
  print "You can repeat the deflation for a total of up to SEVEN steps. After this point,"
  print "the tiles would be very small and the amount of memory required very large, so seven"
  print "steps is the maximum. Note that each higher stage of deflation takes longer to"
  print "compute than its predecessor."
  print ""
  print "To go back and re-choose the starting values, press the HOME key. To quit the program."
  print "press the ESCAPE key,"
  text mm.hres\2, 580, "Press Any Key to Continue", "CT"
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

' Draw a Penrose P2 Tile (Kite or Dart) with the top
' corner of the tile at the specified (x, y) coordinate,
' with the specified scale (length of a long side), with the
' specified rotation (0..9 with 36 degree increment), and with
' the specified edge and fill colors.
'   type: KITE (1) or DART (2)
'   x, y: coordinates of the vertex to which the tile is attached
'   scale: the length of a long side of the tile
'   rot: the tile rotation index 0..9.
'   corner: the tile corner (1..4) that is on the vertex.
'   ec, fc: the edge and fill colors for the tile.
sub DrawP2Tile type, x, y, scale, rot, corner, ec, fc
  local float rangle, l1, l2, a1, ra, sra, dx, dy
  local xv(4), yv(4)
  l1 = scale
  l2 = IG*scale
  ra = 90.0 + rot*36.0
  x1 = x
  y1 = y
  if type = KITE then
    x3 = x + int(l1*cos(ra) + 0.5)
    y3 = y - int(l1*sin(ra) + 0.5)
    a1 = 108.0
  else
    x3 = x + int(l2*cos(ra) + 0.5)
    y3 = y - int(l2*sin(ra) + 0.5)
    a1 = 72.0
  end if
  x2 = x3 + int(l2*cos(a1+ra) + 0.5)
  y2 = y3 - int(l2*sin(a1+ra) + 0.5)
  x4 = x3 + int(l2*cos(a1-ra) + 0.5)
  y4 = y3 + int(l2*sin(a1-ra) + 0.5)
  select case corner
    case 1
      dx = 0
      dy = 0
    case 2
      dx = x2 - x
      dy = y2 - y  
    case 3
      dx = x3 - x
      dy = y3 - y        
    case 4
      dx = x4 - x
      dy = y4 - y
  end select
  xv(1) = x1          : yv(1) = y1
  xv(2) = x2          : yv(2) = y2
  xv(3) = x3          : yv(3) = y3
  xv(4) = x4          : yv(4) = y4
  for i = 1 to 4
    xv(i) = xv(i) - dx
    yv(i) = yv(i) - dy
  next i
  polygon 4, xv(), yv(), ec, fc
end sub

' Read the Vertex info for all 7 vertex types
' Note: the (x, y) pixel coordinates are left zero
sub ReadVertices
  local i, j, k
  for i = 1 to NUM_VERTICES
    read vnum(i)
    for j = 1 to vnum(i)
      for k = 1 to NUM_VINFO
        read vinfo(i, j, k)
      next k
    next j
  next i
end sub 

sub PrintVertices
  local i, j, k
  for i = 1 to NUM_VERTICES
    print #1, "Vertex type " + str$(i) + " has " + str$(vnum(i)) + " tiles"
    for j = 1 to vnum(i)
      print #1, "  Tile " + str$(j)
      print #1, "    (" + str$(vinfo(i, j, 1));
      for k = 2 to NUM_VINFO
        print #1, "," + str$(vinfo(i, j, k));
      next k
      print #1, ")"
    next j
  next i
end sub

' Draw the 7 different vertex types
sub DrawVertices
  local i, j, k, type, corner, rot, cx, cy
  cls
  text MM.HRES\2, 15, "7 Possible Different Arrangements", "CT", 4,, rgb(green)
  text MM.HRES\2, 45, "of Penrose P2 Tiles Around a Vertex", "CT", 4,, rgb(green)
  for i = 1 to NUM_VERTICES
    if i > 3 then
      x = s1 + 3*(i-3)*S1
      y = 400
    else
      x = s1 + 3*i*S1
      y = 200
    end if
    DrawVertex i, x, y, S1
    circle x, y, 3,,, RGB(RED), RGB(RED)
    text x, y+S1+15, vnames$(i), "CT"
  next i
end sub

' Draw a specified filled vertex at the specified location and size
sub DrawVertex vtype, x, y, s
  local i, type, corner, rot, tx, ty
  print #1, "DrawVertex vtype: ";vtype; " x: ";x;" y: ";y; " s: ";s
  for i = 1 to MAX_VTILES
    if vinfo(vtype, i, 3) = 0 then
      exit for
    end if
    tx = vinfo(vtype, i , 1)
    ty = vinfo(vtype, i, 2)
    type = vinfo(vtype, i, 3)
    rot = vinfo(vtype, i, 4)
    corner = vinfo(vtype, i, 5)
    print #1, "  i: ";i;" type: ";type; " tx: ";tx;" ty: ";ty;" rot: ";rot;" corner: ";corner
    DrawP2Tile type, x, y, s, rot, corner, ec, colors(4-type)
  next i
end sub

' 5 unique filled vertices data:
' first line: number of tiles to fill vertex
' next lines: tile type, rotation, and corner that
' attaches to vertex

' Star
data 5
data 0, 0, 2, 1, 1
data 0, 0, 2, 3, 1
data 0, 0, 2, 5, 1
data 0, 0, 2, 7, 1
data 0, 0, 2, 9, 1

' Ace
data 3
data 0, 0, 1, 9, 2
data 0, 0, 2, 5, 3
data 0, 0, 1, 1, 4

' Sun
data 5
data 0, 0, 1, 0, 1
data 0, 0, 1, 2, 1
data 0, 0, 1, 4, 1
data 0, 0, 1, 6, 1
data 0, 0, 1, 8, 1

' King
data 5
data 0, 0, 1, 3, 4
data 0, 0, 2, 8, 1
data 0, 0, 2, 0, 1
data 0, 0, 2, 2, 1
data 0, 0, 1, 7, 2

' Jack
data 5
data 0, 0, 1, 6, 1
data 0, 0, 1, 4, 1
data 0, 0, 2, 9, 4
data 0, 0, 1, 5, 3
data 0, 0, 2, 1, 2

' Queen
data 5
data 0, 0, 1, 9, 2
data 0, 0, 1, 1, 4
data 0, 0, 1, 5, 2
data 0, 0, 2, 0, 1
data 0, 0, 1, 5, 4

' Deuce
data 4
data 0, 0, 1, 2, 3
data 0, 0, 1, 8, 3
data 0, 0, 2, 4, 2
data 0, 0, 2, 6, 4


